/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-06 14:29:28 +0000 (Tue, 06 Feb 2024) $ $Rev: 1787 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.Marshaller;
import jakarta.xml.bind.Unmarshaller;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.SAXException;

import de.consist.bmu.rule.MeldungTyp.FehlerStufe;
import de.consist.bmu.rule.config.schema.fxs.FXSConfig;
import de.consist.bmu.rule.config.schema.fxs.FXSSchemaValidator;
import de.consist.bmu.rule.def.MeldungTypImpl;
import de.consist.bmu.rule.def.RuleDefImpl;
import de.consist.bmu.rule.def.RuleSetDefImpl;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.impl.RuleImpl;
import de.consist.bmu.rule.impl.RuleResultImpl;
import de.consist.bmu.rule.impl.RuleSetImpl;
import de.consist.bmu.rule.impl.RuleSetResultImpl;
import de.consist.bmu.rule.util.ByteUtils;
import de.consist.bmu.rule.xmlsec.XmlSecFassade;

/**
 * Die Factory zum Laden und Speichern von Regeldefinitionen und zum erzeugen von Regelsaetzen.
 */
public final class RuleFactory {
	private static final Log LOGGER = LogFactory.getLog(RuleFactory.class);
	private static final String DEF_RESOURCE_PATH = "/de/consist/bmu/config/";
	private static final String DEFAULT_RULESET_DEF_NAME = RuleFactory.DEF_RESOURCE_PATH + "BMU_Rule_Config_Default.xml";
	private static final String MERGE_XSLT_NAME = "/de/consist/bmu/xslt/merge_ruleconfig.xslt";
	private static File _fxsSchemaBaseDir = new File(System.getProperty("user.dir"));

	private static RuleFactory _theInstance = null;

	/**
	 * Enumersation der Konfigurationsvarianten.
	 */
	public static enum RuleConfig {
		/** ServiceModul. */
		SERVICEMODUL,
		/** Laender-eANV. */
		LEANV,
		/** ASYS. */
		ASYS,
		/** BMU-Viewer. */
		BMUVIEWER,
		/** TEST-Konfiguration ohne Gueltig-Ab und Gueltig-Bis in den Regeln */
		TEST;
		/**
		 * @return Der Name der Resource
		 */
		public String getFileName() {
			return "BMU_Rule_Config_" + this.toString() + ".xml";
		}
	};

	private JAXBContext _jaxbContext = null;
	private Schema _ruleConfigSchema = null;
	private Schema _ruleConfigSchemaFXS = null;
	private RuleConfig _ruleConfig = null;
	private FXSConfig _fxsConfig = null;
	private FXSSchemaValidator _fxsValidator;

	private RuleFactory() {
	}

	private void init() throws BMUException {
		try {
			XmlSecFassade.getInstance();
            // wegen Problemen mit Java-1.8 wieder rausgenommen
			String jaxbContextFactory = System.getProperty(JAXBContext.JAXB_CONTEXT_FACTORY);
            LOGGER.debug(JAXBContext.JAXB_CONTEXT_FACTORY + " : " + jaxbContextFactory);
			jaxbContextFactory = System.getProperty("jakarta.xml.bind.JAXBContextFactory");
            LOGGER.debug("jakarta.xml.bind.JAXBContextFactory : " + jaxbContextFactory);
//            String javaVersion = System.getProperty("java.version");
//            if (jaxbContextFactory == null && !javaVersion.startsWith("1.")) {
//            System.setProperty("jakarta.xml.bind.JAXBContextFactory", "com.sun.xml.bind.v2.ContextFactory");
//			System.setProperty("jakarta.xml.bind.JAXBContextFactory", "org.glassfish.jaxb.runtime.v2.ContextFactory");
//            }
//			ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();

//			try {           
//			    ClassLoader objectFactoryClassLoader = ObjectFactory.class.getClassLoader();            
//			    Thread.currentThread().setContextClassLoader(objectFactoryClassLoader);
				_jaxbContext = JAXBContext
						.newInstance(RuleSetDefImpl.class, RuleDefImpl.class, MeldungTypImpl.class, RuleSetResultImpl.class, RuleResultImpl.class, FXSConfig.class);
//			} finally {
//			    Thread.currentThread().setContextClassLoader(currentClassLoader);
//			}
			SchemaFactory sf = SchemaFactory.newInstance(javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI);
			StreamSource schemaSource = new StreamSource(RuleFactory.class.getResourceAsStream("/de/consist/bmu/rule/config/schema/BMU_Rule_Config.xsd"));
			_ruleConfigSchema = sf.newSchema(schemaSource);
			schemaSource = new StreamSource(RuleFactory.class.getResourceAsStream("/de/consist/bmu/rule/config/schema/fxs/BMU_Rule_Config_FXS.xsd"));
			_ruleConfigSchemaFXS = sf.newSchema(schemaSource);
			//_fxsConfig = FXSConfigDefault.getFXSConfig();
			//_fxsValidator = new FXSSchemaValidator(_fxsConfig.getFXSSchema());
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("error creating JAXBContext", e);
			throw new BMUException("error creating JAXBContext", e);
		} catch (SAXException e) {
			RuleFactory.LOGGER.error("error loading Schema", e);
			throw new BMUException("error loading Schema", e);
		}
	}

	public void loadFXSConfig(File fxsConfigFile, File schemaBaseDir) throws BMUException {
		LOGGER.info("Lade FXS-Konfigurationsdatei: " + fxsConfigFile.getAbsolutePath() + ", Basisverzeichnis fr Schemas: " + ((schemaBaseDir != null) ? schemaBaseDir.getAbsolutePath() : "null"));
		try {
			_fxsConfig = loadFXSConfig(new FileInputStream(fxsConfigFile));
			if (schemaBaseDir != null && schemaBaseDir.exists()) {
				_fxsSchemaBaseDir = schemaBaseDir;
			} else {
			    LOGGER.warn("Basisverzeichnis fr Schemas existiert nicht: " + schemaBaseDir.getAbsolutePath());
			}
			_fxsValidator = new FXSSchemaValidator(_fxsConfig.getFXSSchema());
		} catch (FileNotFoundException e) {
			LOGGER.error("Datei nicht gefunden: " + fxsConfigFile.getAbsolutePath());
			throw new BMUException("FXS-Konfigurationsdatei nicht gefunden: " + fxsConfigFile.getAbsolutePath(), e);
		}
	}
	
	/**
	 * @param is
	 *            InputStream mit FXS-Definitionen
	 * @return FXSConfig
	 * @throws BMUException
	 *             BMUException
	 */
	private FXSConfig loadFXSConfig(InputStream is) throws BMUException {
		FXSConfig fxsConfig = null;
		try {
			Unmarshaller um = _jaxbContext.createUnmarshaller();
			um.setSchema(_ruleConfigSchemaFXS);
			fxsConfig = (FXSConfig) um.unmarshal(is);
			RuleFactory.LOGGER.debug("FXS-Definition geladen: " + fxsConfig);
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error loading FXSConfig from InputStream", e);
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				RuleFactory.LOGGER.warn("closing InputStream failed", e);
			}
		}
		return fxsConfig;
	}

//	/**
//	 * @param os
//	 *            Der OutputStream zum Speichern der FXS-Definitionen
//	 * @param fxsConfig
//	 *            Die FXS-Definition
//	 * @throws BMUException
//	 *             BMUException
//	 */
//	private void saveFXSConfig(OutputStream os, FXSConfig fxsConfig) throws BMUException {
//		try {
//			Marshaller m = _jaxbContext.createMarshaller();
//			m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
//			m.setSchema(_ruleConfigSchemaFXS);
//			m.marshal(fxsConfig, os);
//			RuleFactory.LOGGER.debug("FXS-Definition gesichert: " + fxsConfig);
//		} catch (JAXBException e) {
//			RuleFactory.LOGGER.error("Fehler", e);
//			throw new BMUException("Error saving FXSConfig: " + fxsConfig, e);
//		} finally {
//			try {
//				os.close();
//			} catch (IOException e) {
//				RuleFactory.LOGGER.warn("closing OutputStream failed", e);
//			}
//		}
//	}
	
	public FXSConfig getFXSConfig() {
		return _fxsConfig;
	}
	
	public FXSSchemaValidator getFXSSchemaValidator() {
		return _fxsValidator;
	}
	
	public File getFxsSchemaBaseDir() {
		return _fxsSchemaBaseDir;
	}
	
	/**
	 * @return RuleFactory
	 * @throws BMUException
	 *             BMUException
	 */
	public static synchronized RuleFactory getInstance() throws BMUException {
		if (RuleFactory._theInstance == null) {
			RuleFactory._theInstance = new RuleFactory();
			RuleFactory._theInstance.init();
		}
		return RuleFactory._theInstance;
	}

	/**
	 * @param ruleSetDefFile
	 *            File mit Regeldefinitionen
	 * @return RuleSetDef
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSetDef loadRuleSetDef(File ruleSetDefFile) throws BMUException {
		StreamSource xmlSource = new StreamSource(RuleFactory.class.getResourceAsStream(RuleFactory.DEFAULT_RULESET_DEF_NAME));
		StreamSource xsltSource = new StreamSource(RuleFactory.class.getResourceAsStream(RuleFactory.MERGE_XSLT_NAME));
		RuleSetDef ruleSetDef = null;
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			// File outFile = File.createTempFile("RuleConfig_Merged_", ".xml");
			Transformer tf = TransformerFactory.newInstance().newTransformer(xsltSource);
			tf.setParameter("SRC_PATH", ruleSetDefFile.getAbsolutePath());
			tf.transform(xmlSource, new StreamResult(out));
			ruleSetDef = loadRuleSetDef(new ByteArrayInputStream(out.toByteArray()));
		} catch (Exception e) {
			RuleFactory.LOGGER.error("error loading RuleSetDef", e);
			throw new BMUException("error loading RuleSetDef", e);
		}
		return ruleSetDef;
	}

	/**
	 * @param ruleConfig
	 *            RuleConfig
	 * @return RuleSetDef
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSetDef loadRuleSetDef(RuleConfig ruleConfig) throws BMUException {
		_ruleConfig = ruleConfig;
		File ruleSetDefFile = new File(System.getProperty("java.io.tmpdir"), ruleConfig.getFileName());
		byte[] data = ByteUtils.readFromResource(RuleFactory.DEF_RESOURCE_PATH + ruleConfig.getFileName());
		ByteUtils.writeToFile(ruleSetDefFile.getAbsolutePath(), data);
		RuleSetDef ruleSetDef = loadRuleSetDef(ruleSetDefFile);
		ruleSetDefFile.deleteOnExit();
		return ruleSetDef;
	}

	/**
	 * @return RuleConfig
	 */
	public RuleConfig getRuleConfig() {
		return _ruleConfig;
	}
	
	/**
	 * @return RuleSetDef
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSetDef getDefaultRuleSetDef() throws BMUException {
		return loadRuleSetDef(RuleFactory.class.getResourceAsStream(RuleFactory.DEFAULT_RULESET_DEF_NAME));
	}

	/**
	 * @param is
	 *            InputStream mit Regeldefinitionen
	 * @return RuleSetDef
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSetDef loadRuleSetDef(InputStream is) throws BMUException {
		RuleSetDef ruleSetDef = null;
		try {
			Unmarshaller um = _jaxbContext.createUnmarshaller();
			um.setSchema(_ruleConfigSchema);
			ruleSetDef = (RuleSetDef) um.unmarshal(is);
			RuleFactory.LOGGER.debug("Ruleset-Definition geladen: " + ruleSetDef);
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error loading RuleSetDef from InputStream", e);
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				RuleFactory.LOGGER.warn("closing InputStream failed", e);
			}
		}
		return ruleSetDef;
	}

	/**
	 * @param ruleSetFile
	 *            Die Datei zum Speichern der Regeldefinitionen
	 * @param ruleSetDef
	 *            Die Regeldefinitionen
	 * @throws BMUException
	 *             BMUException
	 */
	public void saveRuleSetDef(File ruleSetFile, RuleSetDef ruleSetDef) throws BMUException {
		try {
			saveRuleSetDef(new FileOutputStream(ruleSetFile), ruleSetDef);
		} catch (FileNotFoundException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error opening file for saving RuleSetDef: " + ruleSetFile.getAbsolutePath(), e);
		}
	}

	/**
	 * @param os
	 *            Der OutputStream zum Speichern der Regeldefinitionen
	 * @param ruleSetDef
	 *            Die Regeldefinitionen
	 * @throws BMUException
	 *             BMUException
	 */
	public void saveRuleSetDef(OutputStream os, RuleSetDef ruleSetDef) throws BMUException {
		try {
			Marshaller m = _jaxbContext.createMarshaller();
			m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			m.setSchema(_ruleConfigSchema);
			m.marshal(ruleSetDef, os);
			RuleFactory.LOGGER.debug("Ruleset-Definition gesichert: " + ruleSetDef);
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error saving RuleSetDef: " + ruleSetDef, e);
		} finally {
			try {
				os.close();
			} catch (IOException e) {
				RuleFactory.LOGGER.warn("closing OutputStream failed", e);
			}
		}
	}

	/**
	 * @param os
	 *            Der OutputStream zum Speichern der Regeldefinitionen
	 * @param ruleSetResult
	 *            Das Pruefergebnis
	 * @throws BMUException
	 *             BMUException
	 */
	public void saveRuleSetResult(OutputStream os, RuleSetResult ruleSetResult) throws BMUException {
		try {
			Marshaller m = _jaxbContext.createMarshaller();
			m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			// m.setSchema(_ruleConfigSchema);
			m.marshal(ruleSetResult, os);
			RuleFactory.LOGGER.debug("Ruleset-Result gesichert: " + ruleSetResult);
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error saving RuleSetResult: " + ruleSetResult, e);
		} finally {
			try {
				os.close();
			} catch (IOException e) {
				RuleFactory.LOGGER.warn("closing OutputStream failed", e);
			}
		}
	}

	/**
	 * @param is
	 *            InputStream mit Regelergebnissen
	 * @return RuleSetResult
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSetResult loadRuleSetResult(InputStream is) throws BMUException {
		RuleSetResult ruleSetResult = null;
		try {
			Unmarshaller um = _jaxbContext.createUnmarshaller();
			// um.setSchema(_ruleConfigSchema);
			ruleSetResult = (RuleSetResult) um.unmarshal(is);
			RuleFactory.LOGGER.debug("Ruleset-Result geladen: " + ruleSetResult);
		} catch (JAXBException e) {
			RuleFactory.LOGGER.error("Fehler", e);
			throw new BMUException("Error loading RuleSetResult from InputStream", e);
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				RuleFactory.LOGGER.warn("closing InputStream failed", e);
			}
		}
		return ruleSetResult;
	}

	/**
	 * Instanziiert aus der RuleSetDefinition ein ausfhrbares RuleSet. Ist das RuleSet wiederverwendbar? Hier knnte man ja auch deaktivierte Regeln
	 * berspringen?!
	 * 
	 * @param ruleSetDef
	 *            Die Regeldefinitionen
	 * @return RuleSet
	 * @throws BMUException
	 *             BMUException
	 */
	public RuleSet createRuleSet(RuleSetDef ruleSetDef) throws BMUException {
		List<Rule> ruleList = new ArrayList<Rule>();
		for (RuleDef ruleDef : ruleSetDef.getRuleList()) {
			RuleImpl ruleImpl = RuleImpl.newRuleImpl(ruleDef);
			if (ruleImpl != null) {
				ruleList.add(ruleImpl);
			} else {
				RuleFactory.LOGGER.error("Error creating Rule from RuleDef: " + ruleDef);
				throw new BMUException("Error creating Rule from RuleDef: " + ruleDef);
			}
		}
		Collections.sort(ruleList, new Comparator<Rule>() {
			@Override
			public int compare(Rule rule1, Rule rule2) {
				if (rule1 instanceof RuleImpl && rule2 instanceof RuleImpl) {
					FehlerStufe fs1 = ((RuleImpl) rule1).getRuleDef().getMeldung().getStufe();
					FehlerStufe fs2 = ((RuleImpl) rule2).getRuleDef().getMeldung().getStufe(); 
					if (fs1.ordinal() > fs2.ordinal()) {
						return 1;
					} else if (fs1.equals(fs2)) {
						return 0;
					} else {
						return -1;
					}
				}
				return 0;
			}
		});
		RuleSet ruleSet = new RuleSetImpl(ruleList, ruleSetDef.getRuleDefRefOK(), ruleSetDef.getRuleDefRefError());
		return ruleSet;
	}

	/**
	 * @param name
	 *            Name der Regel
	 * @return Die technische Dokumentation der Pruefregelimplementierung
	 * @throws BMUException
	 *             BMUException
	 */
	public static String getTechDoc(String name) throws BMUException {
		String techDoc = null;
		try {
			Class<?> ruleClass = Class.forName("de.consist.bmu.rule.impl.RuleImpl" + name);
			Method m = ruleClass.getDeclaredMethod("getTechDoc", (Class<?>[]) null);
			techDoc = (String) m.invoke(null, (Object[]) null);
		} catch (Exception e) {
			RuleFactory.LOGGER.error("error getting techDoc from: " + name, e);
			throw new BMUException("error getting techDoc from: " + name, e);
		}
		return techDoc;
	}
}
