/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-06 14:29:28 +0000 (Tue, 06 Feb 2024) $ $Rev: 1787 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.def;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;
import jakarta.xml.bind.annotation.adapters.XmlAdapter;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import de.consist.bmu.rule.BMUVersion;
import de.consist.bmu.rule.MeldungTyp;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.util.DateUtils;

/**
 * Klasse fuer die Regeldefinition. Wird mit jaxb als xml serialisiert. FIXME
 * Macht die BMU-Version hier Sinn?
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "ruledef", propOrder = { "_version", "_id", "_name",
        "_docTypes", "_enabled", "_keyDate", "_cancel", "_meldung",
        "_beschreibung", "_keyToDate", "_useSignatureDate" })
@XmlRootElement(name = "RuleDef")
public class RuleDefImpl implements RuleDef {

    /** */
    private static final long serialVersionUID = 1L;

    @XmlElement(name = "Version")
    @XmlJavaTypeAdapter(XmlBMUVersionAdapter.class)
    private BMUVersion _version;
    @XmlElement(name = "Id")
    private String _id;
    @XmlElement(name = "Name")
    private String _name;
    @XmlElement(name = "Dokumententypen")
    private List<String> _docTypes;
    @XmlElement(name = "Aktiviert")
    private boolean _enabled;
    @XmlElement(name = "Stichtag")
    @XmlJavaTypeAdapter(XmlDateAdapter.class)
    private Date _keyDate;
    @XmlElement(name = "Abbruch")
    private boolean _cancel;
    @XmlElement(name = "Meldung", type = MeldungTypImpl.class)
    private MeldungTyp _meldung;
    @XmlElement(name = "Info")
    private String _beschreibung;
    @XmlElement(name = "StichtagBis")
    @XmlJavaTypeAdapter(XmlDateAdapter.class)
    private Date _keyToDate;
    @XmlElement(name = "VerwendeSignaturDatum")
    private boolean _useSignatureDate;

    /**
     * Konstruktor.
     */
    public RuleDefImpl() {
    }

    /**
     * @param id
     *            int
     * @param name
     *            String
     * @param docTypes
     *            List
     * @param enabled
     *            boolean
     * @param keyDate
     *            Date
     * @param cancel
     *            boolean
     * @param meldung
     *            MeldungTyp
     * @param beschreibung
     *            String
     */
    public RuleDefImpl(String id, String name, List<String> docTypes,
            boolean enabled, Date keyDate, boolean cancel, MeldungTyp meldung,
            String beschreibung, Date keyToDate, boolean useSignatureDate) {
        this._id = id;
        this._name = name;
        this._docTypes = docTypes;
        this._enabled = enabled;
        this._keyDate = keyDate;
        this._cancel = cancel;
        this._meldung = meldung;
        this._version = BMUVersion.V104;
        this._beschreibung = beschreibung;
        if (keyToDate != null) {
        	this._keyToDate = keyToDate;
        } else {
        	this._keyToDate = new Date(Long.MAX_VALUE);
        }
        this._useSignatureDate = useSignatureDate;
    }

    /**
     * Copy-Konstruktor.
     * 
     * @param source
     *            RuleDef
     */
    public RuleDefImpl(RuleDef source) {
        this._id = source.getId();
        this._name = source.getName();
        if (source.getDocTypes() != null) {
            this._docTypes = new ArrayList<String>();
            for (String docType : source.getDocTypes()) {
                this._docTypes.add(docType);
            }
        }
        this._enabled = source.isEnabled();
        this._keyDate = source.getKeyDate();
        this._cancel = source.isAbbruch();
        this._meldung = new MeldungTypImpl(source.getMeldung());
        this._version = source.getVersion();
        this._beschreibung = source.getBeschreibung();
        this._keyToDate = source.getKeyToDate();
        this._useSignatureDate = source.getUseSignatureDate();
    }

    /**
     * {@inheritDoc}
     */
    public final String getId() {
        return _id;
    }

    /**
     * {@inheritDoc}
     */
    public final void setId(String id) {
        this._id = id;
    }

    /**
     * {@inheritDoc}
     */
    public final String getName() {
        return _name;
    }

    /**
     * {@inheritDoc}
     */
    public final void setName(String name) {
        this._name = name;
    }

    /**
     * {@inheritDoc}
     */
    public final List<String> getDocTypes() {
        return _docTypes;
    }

    /**
     * {@inheritDoc}
     */
    public final void setDocTypes(List<String> docTypes) {
        this._docTypes = docTypes;
    }

    /**
     * {@inheritDoc}
     */
    public final boolean isEnabled() {
        return _enabled;
    }

    /**
     * {@inheritDoc}
     */
    public final void setEnabled(boolean enabled) {
        this._enabled = enabled;
    }

    /**
     * {@inheritDoc}
     */
    public final Date getKeyDate() {
        return _keyDate;
    }

    /**
     * {@inheritDoc}
     */
    public final void setKeyDate(Date keyDate) {
        this._keyDate = keyDate;
    }

    /**
     * {@inheritDoc}
     */
    public final boolean isAbbruch() {
        return _cancel;
    }

    /**
     * {@inheritDoc}
     */
    public final void setAbbruch(boolean cancel) {
        this._cancel = cancel;
    }

    /**
     * {@inheritDoc}
     */
    public final MeldungTyp getMeldung() {
        return _meldung;
    }

    /**
     * {@inheritDoc}
     */
    public final void setMeldung(MeldungTyp meldung) {
        this._meldung = meldung;
    }

    /**
     * {@inheritDoc}
     */
    public final BMUVersion getVersion() {
        return _version;
    }

    /**
     * {@inheritDoc}
     */
    public final String getBeschreibung() {
        return _beschreibung;
    }

    /**
     * {@inheritDoc}
     */
    public final void setBeschreibung(String beschreibung) {
        this._beschreibung = beschreibung;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public final String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Version=" + _version);
        builder.append(", Id=" + _id);
        builder.append(", Name=" + _name);
        builder.append(", Dokumenttypen: " + _docTypes);
        builder.append(", Aktiviert=" + _enabled);
        builder.append(", Stichtag=" + DateUtils.toDateString(_keyDate));
        builder.append(", Abbruch=" + _cancel);
        builder.append(", Meldung: {" + _meldung + "}");
        builder.append(", Beschreibung=" + _beschreibung);
       	builder.append(", StichtagBis=" +  ((_keyToDate != null) ? DateUtils.toDateString(_keyToDate) : "null"));
        builder.append(", VerwendeSignaturDatum=" + _useSignatureDate);
        return builder.toString();
    }

    /**
     * Adapter fuer JAXB.
     */
    public static class Adapter extends XmlAdapter<RuleDefImpl, RuleDef> {
        /**
         * {@inheritDoc}
         */
        @Override
        public final RuleDef unmarshal(RuleDefImpl ruleDefImpl) {
            return ruleDefImpl;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public final RuleDefImpl marshal(RuleDef ruleDef) {
            return (RuleDefImpl) ruleDef;
        }
    }

    /**
     *
     */
    public static class RuleDefComparator implements Comparator<RuleDef>,
            Serializable {
        /**
         * 
         */
        private static final long serialVersionUID = 1L;

        /**
         * {@inheritDoc}
         */
        public final int compare(RuleDef o1, RuleDef o2) {
            if (o1.getMeldung().getStufe().ordinal() < o2.getMeldung()
                    .getStufe().ordinal()) {
                return -1;
            } else if (o1.getMeldung().getStufe().ordinal() > o2.getMeldung()
                    .getStufe().ordinal()) {
                return 1;
            }
            if (o1.isAbbruch()) {
                if (!o2.isAbbruch()) {
                    return -1;
                }
            } else if (o2.isAbbruch()) {
                return 1;
            }
            return 0;
        }

    }

    /**
     * {@inheritDoc}
     */
	public Date getKeyToDate() {
		if (_keyToDate == null) {
			return new Date(Long.MAX_VALUE);
		}
		return _keyToDate;
	}

    /**
     * {@inheritDoc}
     */
	public void setKeyToDate(Date keyToDate) {
		_keyToDate = keyToDate;
	}

    /**
     * {@inheritDoc}
     */
	public boolean getUseSignatureDate() {
		return _useSignatureDate;
	}

    /**
     * {@inheritDoc}
     */
	public void setUseSignatureDate(boolean useSignatureDate) {
		_useSignatureDate = useSignatureDate;
	};
}
