/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-03-28 14:56:24 +0000 (Thu, 28 Mar 2024) $ $Rev: 1800 $
 */
package de.consist.bmu.rule.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXParseException;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUVersion;
import de.consist.bmu.rule.MeldungTyp;
import de.consist.bmu.rule.MeldungTyp.FehlerKlasse;
import de.consist.bmu.rule.MeldungTyp.FehlerStufe;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.RuleSet;
import de.consist.bmu.rule.RuleSetResult;
import de.consist.bmu.rule.def.MeldungTypImpl;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.schema.BMUKopfdaten;
import de.consist.bmu.rule.schema.Namespace;
import de.consist.bmu.rule.schema.SchemaValidator;
import de.consist.bmu.rule.schema.SchemaValidator.ValidationErrorHandler;
import de.consist.bmu.rule.util.DIN_SPEC_91379;
import de.consist.bmu.rule.util.XmlUtils;

/**
 * @author jannighoefer
 * 
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "rulesetresult", propOrder = { "_bmuDokument", "_ruleResultList", "_bezugResult", "_meldungTypList", "_zKSMeldungTypList", "_fehlerStufe", "_status" })
@XmlRootElement(name = "Pruefergebnis")
public final class RuleSetResultImpl implements RuleSetResult, Serializable {

    private static final long serialVersionUID = 1L;

    private static final Log LOGGER = LogFactory
            .getLog(RuleSetResultImpl.class);

    /** UUID fr Fehlerfall (aus dem ServiceModul der ZKS-Abfall). **/
    public static final String UUID_FEHLER = "ABCDEF12-1234-1234-1234-123456789232";

    private transient RuleSet _ruleSet;
    
    @XmlElement(name = "BMUDokument")
    private BMUDokument _bmuDokument;
    @XmlElement(name = "Ergebnisse")
    private List<RuleResult> _ruleResultList;
    @XmlElement(name = "Bezug", type = RuleResultImpl.class)
    private RuleResult _bezugResult;
    @XmlElement(name = "Meldungen")
    private List<MeldungTyp> _meldungTypList = new ArrayList<MeldungTyp>();
    @XmlElement(name = "ZKS-Meldungen")
    private List<MeldungTyp> _zKSMeldungTypList = new ArrayList<MeldungTyp>();
    @XmlElement(name = "Fehlerstufe")
    private FehlerStufe _fehlerStufe;
    @XmlElement(name = "Status")
    private Status _status;

    /**
     * Default Konstruktor. 
     */
    public RuleSetResultImpl() {
    }

    public RuleSetResultImpl(FehlerStufe stufe) {
        _fehlerStufe = stufe;
        _ruleResultList = new ArrayList<RuleResult>();
    }

    /**
     * @param bmuDok
     *            BMUDokument
     * @param ruleSet
     *            Das RuleSet
     * @param list
     *            List
     * @param bezugResult
     *            Das RuleResult mit der Bezugsmeldung
     * @param fehlerStufe
     *            Die hoechste FehlerStufe
     * @param status
     *            Der Status der Pruefung
     */
    public RuleSetResultImpl(BMUDokument bmuDok, RuleSet ruleSet,
            List<RuleResult> list, RuleResult bezugResult,
            FehlerStufe fehlerStufe, Status status) {
        super();
        _bmuDokument = bmuDok;
        _ruleSet = ruleSet;
        _ruleResultList = list;
        _bezugResult = bezugResult;
        _fehlerStufe = fehlerStufe;
        _status = status;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> getList() {
        return _ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public BMUDokument getBMUDokument() {
        return _bmuDokument;
    }

    /**
     * {@inheritDoc}
     */
    public RuleSet getRuleSet() {
        return _ruleSet;
    }

    /**
     * {@inheritDoc}
     */
    public void addMeldung(String klasse, String stufe, String beschreibung,
            String code, String abhilfe) {
        MeldungTyp meldung = new MeldungTypImpl(FehlerKlasse.valueOf(klasse),
                FehlerStufe.valueOf(stufe), beschreibung, code, abhilfe);
        _meldungTypList.add(meldung);
        if (meldung.getStufe().ordinal() < _fehlerStufe.ordinal()) {
            if (FehlerStufe.INFO.equals(_fehlerStufe)) {
                _bezugResult = new RuleResultImpl(
                        _ruleSet.getRuleDefBezugError(), 1,
                        _bezugResult.getMessage());
            }
            _fehlerStufe = meldung.getStufe();
        }
    }

	@Override
	public void addZKSMeldung(String klasse, String stufe, String beschreibung, String code, String abhilfe) {
        MeldungTyp meldung = new MeldungTypImpl(FehlerKlasse.valueOf(klasse),
                FehlerStufe.valueOf(stufe), beschreibung, code, abhilfe);
        _zKSMeldungTypList.add(meldung);
        // Nicht fuer die Ermittlung der hchsten Fehlerstufe beruecksichtigen!
//        if (meldung.getStufe().ordinal() < _fehlerStufe.ordinal()) {
//            if (FehlerStufe.INFO.equals(_fehlerStufe)) {
//                _bezugResult = new RuleResultImpl(
//                        _ruleSet.getRuleDefBezugError(), 1,
//                        _bezugResult.getMessage());
//            }
//            _fehlerStufe = meldung.getStufe();
//        }
	}

    /**
     * {@inheritDoc}
     */
    public RuleResult getBezugResult() {
        return _bezugResult;
    }

    /**
     * {@inheritDoc}
     */
    public FehlerStufe getHoechsteFehlerstufe() {
        return _fehlerStufe;
    }

    /**
     * {@inheritDoc}
     */
    public Status getStatus() {
        return _status;
    }

    /**
     * {@inheritDoc}
     */
    public Document toQuittung() throws BMUException {
        // Hier mssen Informationen aus der empfangenen Nachricht geholt werden
        Document docQuittung = null;
        docQuittung = XmlUtils.newDocument(true);
        Element quittung = docQuittung.createElementNS(
                Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix()
                        + ":Quittung");
        quittung.setAttributeNS(Namespace.TypenBibliothek.getUri(),
                Namespace.TypenBibliothek.getPrefix()
                        + ":Spezifikationsversion",
                BMUVersion.V104.getVersion());
        String bezugMsgUUID = null;
        if (_bmuDokument != null) { 
            bezugMsgUUID = _bmuDokument.getMessageType().getMsgUUID();
        }
        if (bezugMsgUUID == null || bezugMsgUUID.length() == 0) {
            // Hier ist keine BezugMsgUUID (z.B. weil nicht schemavalide oder
            // nur Nutzdaten), also muss die Quittung entsprechend markiert
            // werden: Entweder mit UUID_FEHLER oder zustzlicher Meldung.
//            LOGGER.debug("Erzeuge neue BezugMsgUUID fr Quittung");
            // bezugMsgUUID = UUID.randomUUID().toString();
            bezugMsgUUID = UUID_FEHLER;
        }
        quittung.setAttributeNS(Namespace.Nachricht.getUri(),
                Namespace.Nachricht.getPrefix() + ":BezugMsgUUID", bezugMsgUUID);
        MeldungTypImpl bezugMeldung = (MeldungTypImpl) _bezugResult
                .getRuleDef().getMeldung();
        Element bezug = bezugMeldung.toXml(docQuittung,
                _bezugResult.getMessage(), _bezugResult.getIndex());
        quittung.appendChild(bezug);
        for (RuleResult ruleResult : _ruleResultList) {
            MeldungTypImpl meldungTyp = (MeldungTypImpl) ruleResult
                    .getRuleDef().getMeldung();
            String msg = ruleResult.getMessage();
            if ("SchemaValidationDT".equals(ruleResult.getRuleDef().getName())) {
                LOGGER.debug("Unzulaessige Zeichen im Meldungstext werden durch 'X' ersetzt.");
                msg = DIN_SPEC_91379.replace(msg, 'X');
            }
            Element meldung = meldungTyp.toXml(docQuittung,
                    msg, ruleResult.getIndex());
            quittung.appendChild(meldung);
        }
        int index = 1;
        for (MeldungTyp meldungTyp : _meldungTypList) {
            Element meldung = ((MeldungTypImpl) meldungTyp).toXml(docQuittung,
                    null, index++);
            quittung.appendChild(meldung);
        }
        BMUKopfdaten bmuKopfdaten = null;
        if (_bmuDokument != null && _bmuDokument.getMessageType().isMessage()) {
            bmuKopfdaten = BMUKopfdaten
                    .getKopfdaten(_bmuDokument.getDocument());
            if (bmuKopfdaten.getEmpfaenger() != null) {
                Element nachricht = docQuittung.createElementNS(
                        Namespace.Nachricht.getUri(),
                        Namespace.Nachricht.getPrefix() + ":Nachricht");
                nachricht.setAttributeNS(Namespace.TypenBibliothek.getUri(),
                        Namespace.TypenBibliothek.getPrefix()
                                + ":Spezifikationsversion",
                        BMUVersion.V104.getVersion());
                nachricht.setAttributeNS(Namespace.Nachricht.getUri(),
                        Namespace.Nachricht.getPrefix() + ":MsgUUID", UUID
                                .randomUUID().toString());
                nachricht.setAttributeNS(Namespace.TypenBibliothek.getUri(),
                        Namespace.TypenBibliothek.getPrefix()
                                + ":Nachrichtenstatus", "ORI");
                Element kopfdaten = bmuKopfdaten.toXml(docQuittung);
                nachricht.appendChild(kopfdaten);
                Element nutzdaten = docQuittung.createElementNS(
                        Namespace.Nachricht.getUri(),
                        Namespace.Nachricht.getPrefix() + ":Nutzdaten");
                nachricht.appendChild(nutzdaten);
                nutzdaten.appendChild(quittung);
                docQuittung.appendChild(nachricht);
            } else {
                LOGGER.warn("Der Empfaenger in den Kopfdaten der Nachricht fehlt, Quittung wird als Nutzdatendokument erzeugt.");
                docQuittung.appendChild(quittung);
                // throw new BMUException(
                // "Der Empfaenger in den Kopfdaten der Nachricht fehlt.");
            }
        } else {
            LOGGER.info("Das BMU-Dokument ist nicht vom Typ 'Nachricht', Quittung wird als Nutzdatendokument erzeugt.");
            docQuittung.appendChild(quittung);
        }
        ValidationErrorHandler handler = SchemaValidator.getInstance()
                .validate(docQuittung.getDocumentElement());
        if (handler != null) {
            List<SAXParseException> errorList = handler.getErrorList();
            if (!errorList.isEmpty()) {
                LOGGER.error(
                        "Die erzeugte BMU-Quittung ist nicht schemavalide.",
                        errorList.get(0));
                throw new BMUException(
                        "Die erzeugte BMU-Quittung ist nicht schemavalide.",
                        errorList.get(0));
            }
        } else {
            LOGGER.error("Fehler bei der Schemavalidierung der BMU-Quittung.");
            throw new BMUException(
                    "Fehler bei der Schemavalidierung der BMU-Quittung.");
        }
        return docQuittung;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("\nHchste Fehlerstufe: " + _fehlerStufe.toString());
        sb.append("\nBezug: ");
        sb.append(_bezugResult);
        if (_ruleResultList != null && _ruleResultList.size() > 0) {
            sb.append("\nRuleResults (");
            sb.append(_ruleResultList.size());
            sb.append("):");
            for (RuleResult ruleResult : _ruleResultList) {
                sb.append("\n- [");
                sb.append(ruleResult);
                sb.append("] ");
            }
        }
        if (_meldungTypList != null && _meldungTypList.size() > 0) {
            sb.append("\nZustzliche Meldungen: ");
            for (MeldungTyp meldung : _meldungTypList) {
                sb.append("[");
                sb.append(meldung);
                sb.append("] ");
            }
        }
        return sb.toString();
    }

	@Override
	public List<MeldungTyp> getMeldungList() {
		return _meldungTypList;
	}

    @Override
    public void setFehlerstufe(FehlerStufe stufe) {
        _fehlerStufe = stufe;

    }

	@Override
	public List<MeldungTyp> getZKSMeldungList() {
		return _zKSMeldungTypList;
	}
}
