/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2013-01-19 00:50:18 +0000 (Sat, 19 Jan 2013) $ $Rev: 978 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.schema;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * Diese Klasse dient der Verarbeitung der BMU-Kopfdaten.
 */
public class BMUKopfdaten {
    private static final Log LOGGER = LogFactory.getLog(BMUKopfdaten.class);

    private static final String XPATH_ABSENDER = "/msg:Nachricht/msg:Kopfdaten/msg:Absender";
    private static final String XPATH_EMPFAENGER = "/msg:Nachricht/msg:Kopfdaten/msg:Empfaenger";
    private static final String XPATH_ANTWORT = "/msg:Nachricht/msg:Kopfdaten/msg:Antwort";
    private static final String XPATH_ZUGANGSTYP = "/@lib:Zugangstyp";
    private static final String XPATH_TEXT = "/lib:Text";
    private static final String XPATH_STRUKTUR = "/lib:Struktur";
    private static final String XPATH_BETRIEB = "/lib:BehoerdlicheNummer";
    private static final String XPATH_PROV = "/lib:Provider";
    private static final String XPATH_ZUSATZ = "/lib:Zusatzangaben";
    private static final String XPATH_BEHNR = "/lib:BehoerdlicheNr";
    private static final String XPATH_PRZ = "/lib:Pruefziffer";
    private static final String XPATH_ROLLE = "/@lib:ATBRolle";

    /**
     * Enumeration der Rolle.
     */
    public enum Rolle {
        /** Erzeuger. */
        ERZ("ERZ"),
        /** Befoerderer. */
        BEF("BEF"),
        /** Zwischenlager. */
        ZWL("ZWL"),
        /** Entsorger. */
        ENT("ENT"),
        /** Behoerde. */
        BEH("BEH"),
        /** Bevollmaechtigter des Erzeugers. */
        BEVERZ("BVE"),
        /** Provider. */
        PROV("PRV"),
        /** Makler. */
        MAK("MAK"),
        /** Sonstige. */
        SONST("SNT");
        private String _idRolle;
        Rolle(String idRolle) {
            _idRolle = idRolle;
        }
        /**
         * @return Die Rolle in der Layer- und Signatur-ID
         */
        public String getIDRolle() {
            return _idRolle;
        }
    }

    /**
     * Enumeration des Zugangstyp.
     */
    public enum Zugangstyp {
        /** OSCI. */
        ZKS,
        /** OSCI. */
        OSCI,
        /** EMail. */
        EMAIL,
        /** sonstiger Zugang. */
        SONST,
    }

    private BMUZugang _absender;
    private BMUZugang _empfaenger;
    private BMUZugang _antwort;

    private BehoerdlicheNummer getBehoerdlicheNummer(Document doc, String xPath)
            throws BMUException {
        BehoerdlicheNummer behNummer = null;
        XPathFassade xf = XPathFassade.getInstance();
        Element root = doc.getDocumentElement();
        try {
            if (xf.evalBool(root, "count(" + xPath + ") > 0")) {
                String behNr = xf.evaluate(doc, xPath + XPATH_BEHNR);
                String prz = xf.evaluate(doc, xPath + XPATH_PRZ);
                String r = xf.evaluate(doc, xPath + XPATH_ROLLE);
                Rolle rolle = Rolle.valueOf(r.trim());
                behNummer = new BehoerdlicheNummer(behNr, prz, rolle);
            }
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Auswerten des XPath-Ausdrucks", e);
        }
        return behNummer;
    }

    private BMUZugang getZugang(Document doc, String xPath) throws BMUException {
        BMUZugang zugang = null;
        XPathFassade xf = XPathFassade.getInstance();
        Element root = doc.getDocumentElement();
        try {
            if (xf.evalBool(root, "count(" + xPath + ") > 0")) {
                String typ = xf.evaluate(doc, xPath + XPATH_ZUGANGSTYP);
                Zugangstyp zt = Zugangstyp.OSCI;
                try {
                    zt = Zugangstyp.valueOf(typ.trim());
                } catch (IllegalArgumentException iae) {
                    LOGGER.info("Unbekannter Zugangstyp: " + typ);
                }
                if (xf.evalBool(root, "count(" + xPath + XPATH_TEXT + ") > 0")) {
                    String text = xf.evaluate(doc, xPath + XPATH_TEXT);
                    zugang = new BMUZugangText(zt, text);
                } else {
                    BehoerdlicheNummer btr = getBehoerdlicheNummer(doc, xPath
                            + XPATH_STRUKTUR + XPATH_BETRIEB);
                    BehoerdlicheNummer prov = getBehoerdlicheNummer(doc, xPath
                            + XPATH_STRUKTUR + XPATH_PROV);
                    String zusatz = null;
                    if (xf.evalBool(doc.getDocumentElement(), "count(" + xPath + XPATH_STRUKTUR + XPATH_ZUSATZ + ") > 0")) {
                        zusatz = xf.evaluate(doc, xPath + XPATH_STRUKTUR
                            + XPATH_ZUSATZ);
                    }
                    zugang = new BMUZugangStruktur(zt, btr, prov, zusatz);
                }
            }
        } catch (XPathExpressionException e) {
            throw new BMUException("Fehler beim Auswerten des XPath-Ausdrucks", e);
        }
        return zugang;
    }

    private void init(Document doc) throws BMUException {
        _absender = getZugang(doc, XPATH_ABSENDER);
        _empfaenger = getZugang(doc, XPATH_EMPFAENGER);
        _antwort = getZugang(doc, XPATH_ANTWORT);
    }

    /**
     * @return BMUZugang
     */
    public final BMUZugang getAbsender() {
        return _absender;
    }

    /**
     * @return BMUZugang
     */
    public final BMUZugang getEmpfaenger() {
        return _empfaenger;
    }

    /**
     * @return BMUZugang
     */
    public final BMUZugang getAntwort() {
        return _antwort;
    }

    /**
     * @param doc
     *            Document
     * @return BMUKopfdaten
     * @throws BMUException
     *             BMUException
     */
    public static BMUKopfdaten getKopfdaten(Document doc) throws BMUException {
        BMUKopfdaten kp = new BMUKopfdaten();
        kp.init(doc);
        return kp;
    }

    /**
     * @param doc
     *            Document
     * @return Element
     */
    public final Element toXml(Document doc) {
        Element kopfdaten = null;
        if (getEmpfaenger() != null && getAbsender() != null) {
            kopfdaten = doc.createElementNS(Namespace.Nachricht.getUri(),
                    Namespace.Nachricht.getPrefix() + ":Kopfdaten");
            if (_empfaenger != null) {
                Element absender = doc.createElementNS(
                        Namespace.Nachricht.getUri(),
                        Namespace.Nachricht.getPrefix() + ":Absender");
                absender.setAttributeNS(Namespace.TypenBibliothek.getUri(),
                        Namespace.TypenBibliothek.getPrefix() + ":Zugangstyp",
                        _empfaenger.getZugangstyp().toString());
                absender.appendChild(_empfaenger.toXml(doc));
                kopfdaten.appendChild(absender);
            }
            BMUZugang empfZugang = null;
            if (_antwort != null) {
                empfZugang = _antwort;
            } else if (_absender != null) {
                empfZugang = _absender;
            }
            if (empfZugang != null) {
                Element empfaenger = doc.createElementNS(
                        Namespace.Nachricht.getUri(),
                        Namespace.Nachricht.getPrefix() + ":Empfaenger");
                empfaenger.setAttributeNS(Namespace.TypenBibliothek.getUri(),
                        Namespace.TypenBibliothek.getPrefix() + ":Zugangstyp",
                        Zugangstyp.ZKS.toString());
                empfaenger.appendChild(empfZugang.toXml(doc));
                kopfdaten.appendChild(empfaenger);
            }
        } else {
            // Keine Empfnger und kein Absender
            LOGGER.debug("Keine Empfnger und kein Absender eingetragen");
        }
        return kopfdaten;
    }

    /**
     * Abstrakte Basis-Klasse fuer Zugang.
     */
    public abstract class BMUZugang {
        private Zugangstyp _zugangsTyp;

        BMUZugang(Zugangstyp typ) {
            _zugangsTyp = typ;
        }

        /**
         * @return Zugangstyp
         */
        public final Zugangstyp getZugangstyp() {
            return _zugangsTyp;
        }

        /**
         * @param doc Das Document
         * @return Element
         */
        public abstract Element toXml(Document doc);
    }

    /**
     * Implementierung der Zugangs-Struktur.
     */
    public class BMUZugangStruktur extends BMUZugang {
        private BehoerdlicheNummer _behNummer;
        private BehoerdlicheNummer _provider;
        private String _zusatzAngaben;

        BMUZugangStruktur(Zugangstyp typ, BehoerdlicheNummer btr,
                BehoerdlicheNummer prov, String zusatz) {
            super(typ);
            _behNummer = btr;
            _provider = prov;
            _zusatzAngaben = zusatz;
        }

        /**
         * @return {@link BehoerdlicheNummer}
         */
        public final BehoerdlicheNummer getBehNummer() {
            return _behNummer;
        }

        /**
         * @return {@link BehoerdlicheNummer}
         */
        public final BehoerdlicheNummer getProvider() {
            return _provider;
        }

        /**
         * @return {@link String}
         */
        public final String getZusatzangaben() {
            return _zusatzAngaben;
        }

        /**
         * @param doc
         *            Document
         * @return Element
         */
        @Override
        public final Element toXml(Document doc) {
            Element struktur = doc.createElementNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":Struktur");
            Element behoerdlicheNummer = _behNummer.toXml(doc,
                    "BehoerdlicheNummer");
            struktur.appendChild(behoerdlicheNummer);
            if (_provider != null) {
                Element provider = _provider.toXml(doc, "Provider");
                struktur.appendChild(provider);
            }
            if (_zusatzAngaben != null) {
                Element zusatzangaben = doc.createElementNS(
                        Namespace.TypenBibliothek.getUri(),
                        Namespace.TypenBibliothek.getPrefix()
                                + ":Zusatzangaben");
                zusatzangaben.setTextContent(_zusatzAngaben);
                struktur.appendChild(zusatzangaben);
            }
            return struktur;
        }
    }

    /**
     * Implementierung des Zugang-Text.
     */
    public class BMUZugangText extends BMUZugang {
        private String _text;

        BMUZugangText(Zugangstyp typ, String text) {
            super(typ);
            _text = text;
        }

        /**
         * @return {@link String}
         */
        public final String getText() {
            return _text;
        }

        @Override
        public final Element toXml(Document doc) {
            Element text = doc.createElementNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":Text");
            text.setTextContent(_text);
            return text;
        }
    }

    /**
     * Implementierung der behoerdlichen Nummer.
     */
    public static class BehoerdlicheNummer {
        private String _behNr;
        private String _pz;
        private Rolle _rolle;

        /**
         * @param behNr
         *            String
         * @param pz
         *            String
         * @param rolle
         *            Rolle
         */
        public BehoerdlicheNummer(String behNr, String pz, Rolle rolle) {
            _behNr = behNr;
            _pz = pz;
            _rolle = rolle;
        }

        /**
         * @return String
         */
        public final String getBehNr() {
            return _behNr;
        }

        /**
         * @return String
         */
        public final String getPZ() {
            return _pz;
        }

        /**
         * @return Rolle
         */
        public final Rolle getRolle() {
            return _rolle;
        }

        /**
         * @param doc
         *            Document
         * @param name
         *            String
         * @return Element
         */
        public final Element toXml(Document doc, String name) {
            Element behoerdlicheNummer = doc.createElementNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":" + name);
            behoerdlicheNummer.setAttributeNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":ATBRolle",
                    _rolle.toString());
            Element behoerdlicheNr = doc.createElementNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":BehoerdlicheNr");
            behoerdlicheNr.setTextContent(_behNr);
            behoerdlicheNummer.appendChild(behoerdlicheNr);
            Element pruefziffer = doc.createElementNS(
                    Namespace.TypenBibliothek.getUri(),
                    Namespace.TypenBibliothek.getPrefix() + ":Pruefziffer");
            pruefziffer.setTextContent(_pz);
            behoerdlicheNummer.appendChild(pruefziffer);
            return behoerdlicheNummer;
        }
    };
}
