/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-06 14:29:28 +0000 (Tue, 06 Feb 2024) $ $Rev: 1787 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.MeldungTyp.FehlerStufe;
import de.consist.bmu.rule.BMUMessageTypeEnum;
import de.consist.bmu.rule.Rule;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.RuleSetResult;
import de.consist.bmu.rule.RuleSet;
import de.consist.bmu.rule.Version;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * Diese Klasse implementiert eine Liste von Pruefregeln.
 * 
 * @author srossbroich
 * 
 */
public class RuleSetImpl implements RuleSet, Serializable {
    private static final long serialVersionUID = 1L;
    private static final Log LOGGER = LogFactory.getLog(RuleSetImpl.class);
    private final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat(
            "dd.MM.yyyy HH:mm:ss");
    private List<Rule> _ruleList;
    private RuleDef _ruleDefBezugNoError;
    private RuleDef _ruleDefBezugError;
    private Map<String, Rule> _ruleMap = new HashMap<String, Rule>();
    private Map<String, Rule> _ruleIdMap = new HashMap<String, Rule>();

    /**
     * @param ruleList
     *            Die Liste der Regeln
     * @param ruleDefBezugNoError
     *            Die Regeldefinition fuer die Bezugsmeldung im OK-Fall
     * @param ruleDefBezugError
     *            Die Regeldefinition fuer die Bezugsmeldung im Fehler-Fall
     */
    public RuleSetImpl(List<Rule> ruleList, RuleDef ruleDefBezugNoError,
            RuleDef ruleDefBezugError) {
        this._ruleList = ruleList;
        for (Rule rule : ruleList) {
            if (rule instanceof RuleImpl) {
                _ruleMap.put(((RuleImpl) rule).getRuleDef().getName(), rule);
                _ruleIdMap.put(((RuleImpl) rule).getRuleDef().getId(), rule);
            }
        }
        this._ruleDefBezugNoError = ruleDefBezugNoError;
        this._ruleDefBezugError = ruleDefBezugError;
    }

    /**
     * {@inheritDoc}
     */
    public final RuleSetResult execute(BMUDokument bmuDok) throws BMUException {
        return execute(bmuDok, true);
    }

    /**
     * {@inheritDoc}
     */
    public final RuleSetResult execute(BMUDokument bmuDok,
            boolean pruefeNachricht) throws BMUException {
        List<RuleResult> resultList = new ArrayList<RuleResult>();
        RuleSetResult.Status status = RuleSetResult.Status.OK;
        FehlerStufe fehlerStufe = FehlerStufe.INFO;
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Fhre " + _ruleList.size() + " Prfungen aus.");
        }
        boolean signaturesVerified = false;
        for (Rule rule : _ruleList) {
            if (rule instanceof RuleImpl) {
                if (rule instanceof RuleImplSchemaNachricht && !pruefeNachricht) {
                    LOGGER.debug("Prfung auf 'msg:Nachricht' wird nicht ausgefhrt!");
                    continue;
                }
//                if (rule instanceof RuleImplSchemaValidation) {
//                   continue;
//                }
                RuleImpl ruleImpl = (RuleImpl) rule;
                List<RuleResult> ruleResultList = ruleImpl
                        .executeInternal(bmuDok);
                BMUMessageType msgType = bmuDok.getMessageType();
                if ((rule instanceof RuleImplSignatureVerification && !BMUMessageTypeEnum.RegisterAuszug
                        .equals(msgType.getEnumType()))
                        || (rule instanceof RuleImplSignatureVerificationRA && BMUMessageTypeEnum.RegisterAuszug
                                .equals(msgType.getEnumType()))) {
                    if (ruleImpl.isEnabled(bmuDok)) {
                        signaturesVerified = true;
                        if (!ruleResultList.isEmpty()) {
                            status = RuleSetResult.Status.SIGNATURE_ERROR;
                        }
                    }
                }
                if (!ruleResultList.isEmpty()) {
                    // Darf jedes RuleDef nur eine Meldung erzeugen?
                    // Nein, aber die Fehlerstufe ist bei mehr als einem
                    // Ergebnis immer die Gleiche.
                    FehlerStufe ruleResultFehlerStufe = ruleResultList.get(0)
                            .getRuleDef().getMeldung().getStufe();
                    if (ruleResultFehlerStufe.ordinal() < fehlerStufe.ordinal()) {
                        fehlerStufe = ruleResultFehlerStufe;
                    }
                    if (ruleResultFehlerStufe.equals(FehlerStufe.DEBUG)) {
                    	if (LOGGER.isDebugEnabled()) {
                    		LOGGER.debug("Regel mit Fehlerstufe 'DEBUG' :\n" + ruleResultList);
                    	}
                    } else {
                    	resultList.addAll(ruleResultList);
                    }
                    RuleDef ruleDef = ((RuleImpl) rule).getRuleDef();
                    if (ruleDef.isAbbruch()) {
                        LOGGER.info("Prfung abgebrochen nach Regel: " + ruleDef.getName());
                        break;
                    }
                }
            } else {
                LOGGER.error("Regel erbt nicht von 'RuleImpl': "
                        + rule.getClass().getName());
                throw new BMUException("Regel erbt nicht von 'RuleImpl': "
                        + rule.getClass().getName());
            }
        }
        if (!signaturesVerified) {
            status = RuleSetResult.Status.SIGNATURES_NOT_VERIFIED;
        }
        String docNr = "k.A.";
        if (bmuDok.getMessageType().getDocNr() != null 
                && bmuDok.getMessageType().getDocNr().length() > 0) {
            docNr = bmuDok.getMessageType().getDocNr();
        }
        String bezugMessage = "Fachliche Nummer: "
                + docNr + ", Prfzeitpunkt: "
                + DATE_FORMAT.format(new Date())
                + ", Version der Prfbibliothek: "
                + Version.getVersion();
        RuleResult bezugResult = null;
        if (resultList.isEmpty()) {
            bezugResult = new RuleResultImpl(_ruleDefBezugNoError, 1,
                    bezugMessage);
        } else {
            bezugResult = new RuleResultImpl(_ruleDefBezugError, 1,
                    bezugMessage);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(resultList.size()
                    + " Meldungen erzeugt, hchste Stufe: " + fehlerStufe + ".");
        }
        RuleSetResult rsr = new RuleSetResultImpl(bmuDok, this, resultList, bezugResult,
                fehlerStufe, status);
        try {
        	XPathFassade xpf = XPathFassade.getInstance();
			NodeList nl = xpf.evaluateNodeList(bmuDok.getDocument(), "descendant::svc:Meldung");
			for (int i=0; i<nl.getLength(); i++) {
				Node node = nl.item(i);
				if (node instanceof Element) {
					String klasse = xpf.evaluate((Element) node, "descendant::svc:Klasse");
					String stufe = xpf.evaluate((Element) node, "descendant::svc:Stufe");
					String beschreibung = xpf.evaluate((Element) node, "descendant::svc:Beschreibung");
					String code = xpf.evaluate((Element) node, "descendant::svc:Code");
					String abhilfe = xpf.evaluate((Element) node, "descendant::svc:Abhilfe");
					rsr.addZKSMeldung(klasse, stufe, beschreibung, code, abhilfe);
					LOGGER.debug("ZKS-Meldung verarbeitet: Klasse=" + klasse + ", Stufe=" + stufe + ", Beschreibung=" + beschreibung + ", Code=" + code + ", Abhilfe=" + abhilfe);
				}
			}
		} catch (XPathExpressionException e) {
			LOGGER.error("Error evaluating 'svc:Meldung'", e);
		}
        return rsr;
    }

    /**
     * {@inheritDoc}
     */
    public final List<Rule> getRuleList() {
        return _ruleList;
    }

    /**
     * {@inheritDoc}
     */
    @Deprecated
    public final Rule getRuleByName(String name) {
        return _ruleMap.get(name);
    }

    /**
     * {@inheritDoc}
     */
    public final Rule getRuleById(String id) {
        return _ruleIdMap.get(id);
    }

    /**
     * {@inheritDoc}
     */
    public final RuleDef getRuleDefBezugNoError() {
        return _ruleDefBezugNoError;
    }

    /**
     * {@inheritDoc}
     */
    public final RuleDef getRuleDefBezugError() {
        return _ruleDefBezugError;
    }
}
